/****************************************************************************
*                camera.cpp
*
*  This module implements methods for managing the viewpoint.
*
*  from Persistence of Vision(tm) Ray Tracer
*  Copyright 1996-2002 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other 
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file.
*  If POVLEGAL.DOC is not available it may be found online at -
*
*    http://www.povray.org/povlegal.html.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
* $File: //depot/povray/3.5/source/camera.cpp $
* $Revision: 1.3 $
* $Change: 1817 $
* $DateTime: 2002/07/27 10:45:37 $
* $Author: HIV $
* $Log: camera.cpp,v $
*
*
******************************************************************************
*
* This is an unofficial patched Version of POV-Ray 3.5
* STEREO-trace Patch 3/2002 by Hermann Vosseler <Ichthyostega@web.de>
*
*****************************************************************************/

#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "camera.h"
#include "matrices.h"
#include "normal.h"



/*****************************************************************************
*
* FUNCTION
*
*   Translate_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Translate_Camera(CAMERA *Camera, VECTOR Vector)
{
  VAddEq(((CAMERA *)Camera)->Location, Vector);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Rotate_Camera(CAMERA *Camera, VECTOR Vector)
{
  TRANSFORM Trans;
  
  Compute_Rotation_Transform(&Trans, Vector);
  
  Transform_Camera(Camera, &Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Scale_Camera(CAMERA *Camera, VECTOR Vector)
{
  TRANSFORM Trans;
  
  Compute_Scaling_Transform(&Trans, Vector);
  
  Transform_Camera(Camera, &Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Aug 2002 : Stereo_base now subject to transformations as well [8/2002 HIV]
*
******************************************************************************/

void Transform_Camera(CAMERA *Camera, TRANSFORM *Trans)
{
  MTransPoint(Camera->Location, Camera->Location, Trans);
  MTransDirection(Camera->Direction, Camera->Direction, Trans);
  MTransDirection(Camera->Up, Camera->Up, Trans);
  MTransDirection(Camera->Right, Camera->Right, Trans);

  if (fabs(Camera->Stereo_base) > EPSILON)
  {
    VECTOR tmpBase;
    int s = Camera->Stereo_base>=0? 1 : -1; // save sign
    VNormalize(tmpBase,Camera->Right);
    VScaleEq(tmpBase,Camera->Stereo_base);

    // transform the vector representation 
    // of the stereo_base as well; only the
    // length of the base matters though. 
    MTransDirection(tmpBase,tmpBase, Trans);
    VLength(Camera->Stereo_base, tmpBase);
    Camera->Stereo_base *= s;
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_Camera
*
* INPUT
*   
* OUTPUT
*   
* RETURNS
*   
* AUTHOR
*
*   POV-Ray Team
*   
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

CAMERA *Create_Camera()
{
  CAMERA *New;
  
  New = (CAMERA *)POV_MALLOC(sizeof (CAMERA), "camera");
  
  Make_Vector(New->Location,    0.0,  0.0, 0.0);
  Make_Vector(New->Direction,   0.0,  0.0, 1.0);
  Make_Vector(New->Up,          0.0,  1.0, 0.0);
  Make_Vector(New->Right,       1.33, 0.0, 0.0);
  Make_Vector(New->Sky,         0.0,  1.0, 0.0);
  Make_Vector(New->Look_At,     0.0,  0.0, 1.0);
  Make_Vector(New->Focal_Point, 0.0,  0.0, 1.0);

  /* Init focal blur stuff (not used by default). */
  New->Blur_Samples   = 0;
  New->Confidence     = 0.9;
  New->Variance       = 1.0 / 10000.0;
  New->Aperture       = 0.0;
  New->Focal_Distance = -1.0;

  /* Set default camera type and viewing angle. [DB 7/94] */
  New->Type = PERSPECTIVE_CAMERA;
  New->Angle = 90.0;

  /* Default view angle for spherical camera. [MH 6/99] */
  New->H_Angle = 360;
  New->V_Angle = 180;

  /* Do not perturb primary rays by default. [DB 7/94] */
  New->Tnormal = NULL;

  /* STEREO: Do not use stereoscopic tracing by default. */
  New->Stereo_base = 0.0;

  /*       convienience shortcuts not used by default. */
  New->Window_distance = 0.0;
  New->Screen_width    = 0.0;

  
  New->Trans = Create_Transform();

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_Camera
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   POV-Ray Team
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

CAMERA *Copy_Camera(CAMERA *Old)
{
  CAMERA *New;

  if (Old != NULL)
  {
    New = Create_Camera();

    Destroy_Tnormal(New->Tnormal);
    Destroy_Transform(New->Trans);

    *New = *Old;
    New->Tnormal = NULL; // clear in case the copy fails
    if(Old->Tnormal != NULL)
       New->Tnormal = Copy_Tnormal(Old->Tnormal);

    New->Trans = NULL; // clear in case the copy fails
    if(Old->Trans != NULL)
       New->Trans = Copy_Transform(Old->Trans);
  }
  else
  {
    New = NULL;
  }

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Destroy_Camera
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Dieter Bayer
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   -
*
******************************************************************************/

void Destroy_Camera(CAMERA *Camera)
{
  if (Camera != NULL)
  {
    Destroy_Tnormal(Camera->Tnormal);
    Destroy_Transform(Camera->Trans);

    POV_FREE(Camera);
  }
}





/*****************************************************************************
*
* FUNCTION
*
*   Setup_Stereo_Camera
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* AUTHOR
*
*   Hermann Vosseler
*
* DESCRIPTION
*
*   Resolves any "convienience shortcut" parameters
*   and defines the final basic camera parameters including the
*   stereo_base.
*   All Stereo Cameras in this patch are defined in a way producing
*   images with built-in stereoscopic window. The window is located
*   in the projection plane/cylinder/sphere of the given camera, i.e.
*   it is to be found at (location + direction).
*   
*   Setting up stereo base and window location in accordance with the
*   arrangement of the objects in scene can be difficult and requires
*   some experience. For this reason, two "convienience shortcuts"
*   and a heuristic for guessing the stereo_base are provided.
*
*   The guess for the stereo_base assumes a standard image viewing
*   situation, where the image is roughly at 2m distance and the
*   largest square included in the image has a diameter of 2m.
*   It is further assumed, that the presentation of the image is
*   "ortho" (ortho-stereoscopic) i.e. all objects are depicted with
*   natural size and distance. 
*   (This heuristic is sometimes refered to as "1:30" or "1/f" rule)
*
*
*   Any one of the three parameters is sufficient to set up a stereo
*   camera, but all may be provided, depending on the situation. They
*   will allways be applied in fixed order: First, the camera is scaled
*   to yield the window_distance, then the base is set or guessed and
*   finally everything besides the stereo_base is scaled, in order to
*   create a window with given window_width.
*   Note that later on (in Create_Ray() ) only the Camera->Stereo_base
*   is used, i.e. stereo_base and the length of the direction-vector
*   are the real parameters for controlling the stereoscopy.
*
*   The simplest use case is to specify only the distance to the
*   window (in most cases, this is the distance to the nearest object).
*   But I want to encurrage the use of real-world units; in the latter
*   case it is best to only set the stereo_base to the normal human
*   eye-spacing of 65mm
*
* CHANGES
*
*   Aug 2002 : Creation (STEREO-patch for POV-Ray 3.5) [HIV]
*
******************************************************************************/

void Setup_Stereo_Camera (CAMERA *Cam)
{
  DBL f,u,r;
  VLength(f,Cam->Direction);
  VLength(u,Cam->Up);
  VLength(r,Cam->Right);

  if (Cam->Window_distance > EPSILON)
  {
     // make sure the stereoscopic window will be located
    //  at given distance from camera location
    DBL scale = Cam->Window_distance / f;
    VScaleEq(Cam->Direction,scale);
    VScaleEq(Cam->Right,scale);
    VScaleEq(Cam->Up,scale);
    f = Cam->Window_distance;
    u*=scale;
    r*=scale;
  }

  if (fabs(Cam->Stereo_base) <= EPSILON)
  {
     // stereoscopic base definition is missing
    //  provide a heurisitc guess 
    int s = Cam->Stereo_base>=0? 1 : -1;
    DBL q = (r>u)? u : r;
    DBL f_standard = q * sqrt(2);
    Cam->Stereo_base = s * (0.065/2) * f_standard;
  }

  if (Cam->Screen_width > EPSILON)
  {
      // The final image is supposed to have the given width.
     //  To accomodate, we simple scale the whole universe....
    //   Rather, we inverse scale the stereo_base :-)
    Cam->Stereo_base *= 1 / (Cam->Screen_width / r);
  }

}
